<?php
namespace Forums\Models;
use App\Models\User;
use Forums\Models\ForumPermission;
use Forums\Models\ForumCategory;

class ForumUser extends User
{
    public $table = 'users';

    public function posts() {
        return $this->hasMany(ForumPost::class, 'user_id');
    }

    public function getPostsPerDayAttribute() {
        $days_since = date_diff(date_create($this->created),date_create(date("Y-m-d H:i:s")))->format("%a") ?? 1;
        if ($days_since < 1) { $days_since = 1; }
        return $this->posts->count() / $days_since;
    }

    public function hasForumPermission($name, $column, $value) {
        $permission = ForumPermission::where('name', $name);

        if ($this->isBannedFromWeb()) {
            $permission = $permission->where('subject', 'banned');
        } else {
            $permission =  $permission->where(function($query) {
                $query->whereIn('rid', $this->roles->pluck('rid'))->orWhere('subject', 'everyone');
            });
        }

        if ($column != null) {
            $permission = $permission->where($column, $value);
        }

        if ((clone $permission)->where('type', 'allow')->exists()) {
            return 'allow'; // allow overrides denies from other roles
        } else {
            if ($permission->where('type', 'deny')->exists()) {
                return 'deny';
            }
        }
    }

    public function hasForumPermissionInBoard($name, $board) {
        // forum permissions < category permissions < board permissions
        $boardPermission = $this->hasForumPermission($name, 'bid', $board->bid);
        if ($boardPermission == 'allow') {
            return true;
        } elseif ($boardPermission == 'deny') {
            return false;
        } else {
            $categoryPermission = $this->hasForumPermission($name, 'cid', $board->category->cid);
            if ($categoryPermission == 'allow') {
                return true;
            } elseif ($categoryPermission == 'deny') {
                return false;
            } else {
                return $this->hasPermission('forums_'.$name);
            }
        }
    }

    public function hasForumPermissionInThread($name, $thread) {
        return $this->hasForumPermissionInBoard($name, $thread->board);
    }

    public function canModerate($thread) {
        return $this->hasForumPermissionInThread('moderate', $thread);
    }

    public function canPostThread($board) {
        return $this->hasForumPermissionInBoard('post_thread', $board);
    }

    public function canPostReply($thread) {
        return $this->hasForumPermissionInThread('post_reply', $thread);
    }

    public function canEditPost($thread) {
        return $this->hasForumPermissionInThread('edit', $thread);
    }

    public function canReact($thread) {
        return $this->hasForumPermissionInThread('react', $thread);
    }

    public function viewableCategories() {
        if ($this->isBannedFromWeb()) {
            return ForumCategory::disableCache()->whereDoesntHave('permissions', function ($query) {
                $query->where(['name' => 'view', 'type' => 'deny'])->where('subject', 'banned');
            })->get();
        } else {
            return ForumCategory::disableCache()->whereHas('permissions', function ($query) {
                $query->where(['name' => 'view', 'type' => 'allow'])->where(function($query) {
                    $query->whereIn('rid', $this->roles->pluck('rid'))->orWhere('subject', 'everyone');
                });
            })->orWhereDoesntHave('permissions', function ($query) {
                $query->where(['name' => 'view'])->where(function($query) {
                    $query->whereIn('rid', $this->roles->pluck('rid'))->orWhere('subject', 'everyone');
                });
            })->get();
        }
    }

    public function viewableBoards() {
        if ($this->isBannedFromWeb()) {
            return ForumBoard::disableCache()->whereIn('cid', $this->viewableCategories()->pluck('cid'))->whereDoesntHave('permissions', function ($query) {
                $query->where(['name' => 'view', 'type' => 'deny'])->where('subject', 'banned');
            })->orWhereHas('permissions', function ($query) {
                $query->where(['name' => 'view', 'type' => 'allow'])->where('subject', 'banned');
            })->get();
        } else {
            return ForumBoard::disableCache()->whereIn('cid', $this->viewableCategories()->pluck('cid'))->whereDoesntHave('permissions', function ($query) {
                $query->where(['name' => 'view', 'type' => 'deny'])->where(function($query) {
                    $query->whereIn('rid', $this->roles->pluck('rid'))->orWhere('subject', 'everyone');
                });
            })->orWhereHas('permissions', function ($query) {
                $query->where(['name' => 'view', 'type' => 'allow'])->where(function($query) {
                    $query->whereIn('rid', $this->roles->pluck('rid'))->orWhere('subject', 'everyone');
                });
            })->get();
        }
    }
}
