<?php
namespace Forums\Controllers;

use App\Controllers\Controller;
use Illuminate\Database\Capsule\Manager as DB;
use App\Models\User;
use Forums\Models\ForumThread;
use Forums\Models\ForumPost;
use Forums\Models\ForumReaction;
use Forums\Models\ForumUser;
use App\Models\Setting;
use League\HTMLToMarkdown\HtmlConverter;
use Forums\Models\ForumCategory;
use Forums\Models\ForumBoard;
use Forums\Models\ForumThreadsRead;
use App\Helpers;

class ForumsApiController extends Controller
{
    public function getIndex($request, $response) {
        $forumCategories = ForumCategory::whereHasViewableBoards($this->auth)->get();
        $viewableBoards = ForumBoard::getViewable($this->auth);

        foreach ($forumCategories as $fC) {
            $boards = $fC->boards()->whereIn('bid', $viewableBoards->pluck('bid'))->with(['latest_thread','latest_thread.read_timestamp_relation','latest_thread.latest_post.user.group'])->get();

            foreach($boards as $board) {
                $board->total_threads = $board->total_threads;
                $board->total_posts = $board->total_posts;

                if (isset($board->latest_thread)) {
                    $board->latest_thread->latest_post->user->append('avatar');
                    $board->latest_thread->read_timestamp = $board->latest_thread->read_timestamp;
                }
            }

            $fC->boards = $boards;
        }

        $latestPosts = ForumPost::whereHas('thread', function ($query) use ($viewableBoards) {
            $query->whereIn('bid', $viewableBoards->pluck('bid'));
        })->orderBy('timestamp', 'DESC')->with(['user.group','thread.read_timestamp_relation'])->take(5)->get();

        foreach ($latestPosts as $lP) {
            $lP->user->append('avatar');
            $lP->thread->read_timestamp = $lP->thread->read_timestamp;
        }

        return $response->withJSON([
          'forum_categories' => Helpers::whitelist_keys(json_decode(json_encode($forumCategories), true), [
              '*' => [
                  'cid',
                  'name',
                  'boards' => [
                      '*' => [
                          'bid',
                          'name',
                          'icon',
                          'total_threads',
                          'total_posts',
                          'latest_thread' => [
                              'topic',
                              'last_posted',
                              'last_post_user_id',
                              'read_timestamp',
                              'latest_post' => [
                                  'pid',
                                  'timestamp',
                                  'user' => [
                                      'id',
                                      'name',
                                      'avatar',
                                      'group' => [
                                          'color'
                                      ]
                                  ]
                              ]
                          ]
                      ]
                  ]
              ]
          ]),
          'latest_posts' => Helpers::whitelist_keys(json_decode(json_encode($latestPosts), true), [
              '*' => [
                  'pid',
                  'tid',
                  'content',
                  'timestamp',
                  'last_edit',
                  'user' => [
                      'id',
                      'name',
                      'avatar',
                      'group' => [
                          'color'
                      ]
                  ],
                  'thread' => [
                      'topic',
                      'last_posted',
                      'read_timestamp'
                  ]
              ]
          ]),
          'forum_statistics' => [
              'total_posts' => ForumPost::count(),
              'total_threads' => ForumThread::count(),
              'total_users' => User::count()
          ]
        ]);
    }

    public function getSearch($request, $response, $args) {
        $params = $request->getParams();
        $perPage = 25;

        $viewableBoards = ForumBoard::getViewable($this->auth);

        $threads = ForumThread::disableCache()->whereIn('bid', $viewableBoards->pluck('bid'))->where(function ($query) use ($params) {
            $query->where('topic', 'like', "%{$params['keyword']}%")->orWhereHas('first_post', function ($query) use ($params) {
                $query->where('content', 'like', "%{$params['keyword']}%");
            })->orWhereHas('user', function ($query) use ($params) {
                $query->where('name', 'like', "%{$params['keyword']}%");
            });
        });

        $threads = $threads->with(['user','user.group','latest_post','latest_post.user','latest_post.user.group','read_timestamp_relation'])->orderBy('pinned','DESC')->orderBy('last_posted','DESC')->paginate($perPage, ['*'], 'page', $params['page'] ?? 1);
        foreach ($threads as $thread) {
            $thread->postcount = $thread->posts()->count();
            $thread->user->append('avatar');
            if ($thread->latest_post) {
                $thread->latest_post->user->append('avatar');
            }
            $thread->read_timestamp = $thread->read_timestamp;
        }

        return $response->withJSON([
          'threads' => Helpers::whitelist_keys(json_decode(json_encode($threads), true), [
            'current_page',
            'last_page',
              'data' => [
                  '*' => [
                      'tid',
                      'bid',
                      'user_id',
                      'topic',
                      'timestamp',
                      'last_posted',
                      'locked',
                      'pinned',
                      'postcount',
                      'read_timestamp',
                      'user' => [
                          'id',
                          'name',
                          'avatar',
                          'group' => [
                              'color'
                          ]
                      ],
                      'latest_post' => [
                          'timestamp',
                          'user' => [
                              'id',
                              'name',
                              'avatar',
                              'group' => [
                                  'color'
                              ]
                          ]
                      ]
                  ]
              ]
          ])
        ]);
    }

    public function getBoard($request, $response, $args) {
      $perPage = 25;
      $board = ForumBoard::find($args['bid']);

      if ($board == null) {
          return $response->withStatus(404);
      }

      $threads = ForumThread::where('bid', $args['bid'])->with(['user','user.group','latest_post','latest_post.user','latest_post.user.group','read_timestamp_relation'])->orderBy('pinned','DESC')->orderBy('last_posted','DESC')->paginate($perPage, ['*'], 'page', $args['page'] ?? 1);
      foreach ($threads as $thread) {
          $thread->postcount = $thread->posts()->count();
          $thread->user->append('avatar');
          if ($thread->latest_post) {
              $thread->latest_post->user->append('avatar');
          }
          $thread->read_timestamp = $thread->read_timestamp;
      }

      $totalPages = ceil($threads->count()/$perPage) ?? 1;

      return $response->withJSON([
        'board' => $board,
        'threads' => Helpers::whitelist_keys(json_decode(json_encode($threads), true), [
          'current_page',
          'last_page',
            'data' => [
                '*' => [
                    'tid',
                    'bid',
                    'user_id',
                    'topic',
                    'timestamp',
                    'last_posted',
                    'locked',
                    'pinned',
                    'postcount',
                    'read_timestamp',
                    'user' => [
                        'id',
                        'name',
                        'avatar',
                        'group' => [
                            'color'
                        ]
                    ],
                    'latest_post' => [
                        'timestamp',
                        'user' => [
                            'id',
                            'name',
                            'avatar',
                            'group' => [
                                'color'
                            ]
                        ]
                    ]
                ]
            ]
        ]),
        'can_post_thread' => $this->auth->check() ? ForumUser::find($this->auth->id())->canPostThread($board): false
      ]);
    }

    public function getThread($request, $response, $args) {
        $perPage = 15;
        $thread = ForumThread::find($args['tid']);

        if ($thread == null) {
            return $response->withStatus(404);
        }

        $posts = $thread->posts()->with(['reactions','quotedPost','quotedPost.user','quotedPost.user.group','user','user.group'])->paginate($perPage, ['*'], 'page', $args['page'] ?? 1)->keyBy('pid');
        foreach($posts as $post) {
            $post->setRelation('reactions', $post->reactions->groupBy('rname'));
            $post->user->append('avatar');
            if ($post->quotedPost)
                $post->quotedPost->user->append('avatar');
            $post->user->post_count = $post->user->posts()->count();
        }

        $totalPages = ceil($thread->posts()->count()/$perPage) ?? 1;

        if ($this->auth->check()) {
            $user = ForumUser::find($this->auth->id());
            ForumThreadsRead::updateOrCreate(['user_id' => $user->id, 'tid' => $args['tid']], ['timestamp' => DB::raw('NOW()')])->flushCache();
        }

        return $response->withJSON([
          'board' => ForumBoard::find($thread->bid),
          'thread' => $thread,
          'posts' => Helpers::whitelist_keys(json_decode(json_encode($posts), true), [
              '*' => [
                  'pid',
                  'reply_to_pid',
                  'content',
                  'timestamp',
                  'last_edit',
                  'reactions',
                  'quoted_post' => [
                      'pid',
                      'content',
                      'user' => [
                          'id',
                          'name',
                          'avatar',
                          'group' => [
                              'color',
                          ]
                      ]
                  ],
                  'user' => [
                      'id',
                      'name',
                      'avatar',
                      'status',
                      'post_count',
                      'group' => [
                          'name',
                          'color',
                          'icon'
                      ]
                  ]
              ]
          ]),
          'reactions_enabled' => Setting::find('reactions_enabled')->value??false,
          'can_moderate' => ($user ?? false) ? $user->canModerate($thread): false,
          'can_post_reply' => ($user ?? false) ? $user->canPostReply($thread): false,
          'can_edit_post' => ($user ?? false) ? $user->canEditPost($thread): false,
          'pagination' => [
              'current' => $args['page'] ?? 1,
              'total' => $totalPages
          ]
        ]);
    }

    public function getPost($request, $response, $args) {
        $perPage = 15;
        $post = ForumPost::find($args['pid']);
        if ($post == null) {
            return $response->withStatus(404);
        }
        $thread = $post->thread;

        $postIndex = 1;
        foreach ($thread->posts as $post) {
            if ($args['pid'] == $post->pid) {
                $page = ceil($postIndex/$perPage);
                break;
            }
            $postIndex++;
        }

        return $response->withJSON([
          'tid' => $thread->tid,
          'page' => $page
        ]);
    }

    static function contentToMarkdown($content) {
        $converter = new HtmlConverter([
            'strip_tags' => true,
            'hard_break' => true,
            'header_style' => 'atx'
        ]);
        $mdContent = $converter->convert($content);
        return str_replace('<pre class="ql-syntax" spellcheck="false">', '', $mdContent); // workaround for https://github.com/thephpleague/html-to-markdown/issues/130
    }

    public function create($request, $response, $args) {
        $params = $request->getParams();
        $user = ForumUser::find($this->auth->id());

        if (!$user->canPostThread(ForumBoard::find($params['bid'])))
            throw new \Slim\Exception\HttpForbiddenException($request);

        $thread = ForumThread::create([
            'bid' => $params['bid'],
            'user_id' => $this->auth->id(),
            'topic' => $params['topic']
        ]);

        ForumPost::create([
            'tid' => $thread->tid,
            'user_id' => $this->auth->id(),
            'content' => self::contentToMarkdown($params['content'])
        ]);
        $thread->flushCache();

        return $response->withJSON(['status' => 'success','tid' => $thread->tid]);
    }

    public function reply($request, $response, $args) {
        $params = $request->getParams();
        $user = ForumUser::find($this->auth->id());
        $thread = ForumThread::find($params['tid']);
        $bid = $thread->bid;

        if (!$user->canPostReply($thread)) { throw new \Slim\Exception\HttpForbiddenException($request); }

        $post = ForumPost::create([
            'tid' => $params['tid'],
            'user_id' => $this->auth->id(),
            'content' => self::contentToMarkdown($params['content']),
            'reply_to_pid' => is_numeric($params['reply_to_pid'])? $params['reply_to_pid']: null
        ]);
        $post->flushCache();

        $thread->update(['last_posted' => DB::raw('NOW()')]);

        return $response->withJSON(['status' => 'success', 'pid' => $post->pid, 'page' => $post->page]);
    }

    public function patchThread($request, $response, $args) {
        $params = $request->getParams();
        $user = ForumUser::find($this->auth->id());
        $thread = ForumThread::find($args['tid']);

        if (!$user->canEditPost($thread)) throw new \Slim\Exception\HttpForbiddenException($request);

        if ($thread->user_id == $user->id) {
            $thread->update(['topic' => $params['topic']]);
            return $response->withJSON(['status' => 'success']);
        } else {
            throw new \Slim\Exception\HttpForbiddenException($request);
        }
    }

    public function edit($request, $response, $args) {
        $params = $request->getParams();
        $user = ForumUser::find($this->auth->id());
        $post = ForumPost::find($params['pid']);

        if (!$user->canEditPost($post->thread)) { throw new \Slim\Exception\HttpForbiddenException($request); }

        if ($post->user_id == $user->id) { // TODO: allow moderators to edit posts from other users (and log edits)
            $post->update(['content' => self::contentToMarkdown($params['content'])]);
            return $response->withJSON(['status' => 'success', 'pid' => $post->pid]);
        } else {
            throw new \Slim\Exception\HttpForbiddenException($request);
        }
    }

    public function delete($request, $response, $args) {
        $params = $request->getParams();
        $user = ForumUser::find($this->auth->id());
        $post = ForumPost::find($params['pid']);

        if ($user->id !== $post->user_id && !$user->canModerate($post->thread)) {
            throw new \Slim\Exception\HttpForbiddenException($request);
        }

        $thread = ForumThread::find($params['tid']);

        if ($post->pid === $thread->first_post->pid) {
            $thread->delete();
        } else {
            $post->delete();
        }

        return $response->withJSON(['status' => 'success']);
    }

    public function mentionAutoComplete($request, $response, $args) {
        $params = $request->getParams();
        $likeStr = '%'.str_replace('@','',$params['matchStr']).'%';
        $users = ForumUser::where('name', 'like', $likeStr)->take(10)->get();
        $users->map->append('avatar');
        return $response->withJSON(['status' => 'success','results' => $users]);
    }

    public function react($request, $response, $args) {
        $params = $request->getParams();
        $user = ForumUser::find($this->auth->id());

        $post = ForumPost::find($params['pid']);

        $tid = $post->tid;
        $thread = ForumThread::find($tid);
        $bid = $thread->bid;

        if (!$user->canReact($thread)) throw new \Slim\Exception\HttpForbiddenException($request);

        $settings = Setting::where('category','forums')->get()->keyBy('setting')->toArray();
        if ($settings['reactions_enabled']['value'] && $post->user_id != $user->id) {
            $reaction = ForumReaction::where(['user_id' => $user->id])->where(['rname' => $params['rname']])->where(['pid' => $params['pid']])->first();
            if ($reaction == null) {
                $count = ForumReaction::where(['user_id' => $user->id])->where(['pid' => $params['pid']])->count();
                if ($count < $settings['max_reactions_per_user_per_post']['value']) {
                    $reaction = ForumReaction::create(['user_id' => $user->id,'rname' => $params['rname'], 'pid' => $params['pid']]);
                    $reaction->flushCache();
                } else {
                    return $response->withJSON(['status' => 'max_reactions_reached']);
                }
            } else {
                $reaction->delete();
            }
        }
        return $response->withJSON(['status' => 'success', 'reactions' => ForumReaction::where(['pid' => $params['pid']])->get()->keyBy('rname') ?? null]);
    }

    public function toggleThreadState($request, $response, $args) {
        $params = $request->getParams();
        $newVal = null; if ($params['val'] == 1) { $newVal = 1; }
        ForumThread::where('tid', $params['tid'])->update([$params['state'] => $newVal]);
        return $response->withJSON(['status' => 'success']);
    }

    public function moveThread($request, $response, $args) {
        $params = $request->getParams();
        ForumThread::where('tid', $params['tid'])->update(['bid' => $params['bid']]);
        return $response->withJSON(['status' => 'success']);
    }
}
