<?php
namespace StripeIntegration\Controllers;

use Stripe\Stripe;
use Stripe\Checkout\Session as StripeSession;
use Slim\Routing\RouteContext;
use App\Controllers\Controller;
use App\Models\Setting;
use App\Models\StorePayment;
use App\Store;

class StripeController extends Controller
{
    public function session($request, $response, $args) {
        $settings = Setting::getByCategory('store');
        $params = $request->getParams();
        $total = $request->getAttribute('total');

        $metadata = [
            'user_id' => $this->auth->id(),
            'type' => $params['type'],
            'total' => $total
        ];

        if ($params['type'] == 'credits') {
            $metadata['quantity'] = $params['quantity'];
            $lineItem = [
                'name' => 'Credits',
                'amount' => $total / $params['quantity'],
                'currency' => $settings['currency'],
                'quantity' => $params['quantity']
            ];
        } else if ($params['type'] == 'package') {
            $package = $request->getAttribute('package');
            $metadata['package_id'] = $package->id;
            $lineItem = [
                'name' => $package->name,
                'amount' => $total,
                'currency' => $settings['currency'],
                'quantity' => 1
            ];
        }

        Stripe::setApiKey($settings['stripe_secret_key']);
        $returnUrl = RouteContext::fromRequest($request)->getRouteParser()->fullUrlFor($request->getUri(), 'store_purchases');

        $session = StripeSession::create([
            'payment_method_types' => ['card'],
            'line_items' => [ $lineItem ?? null ],
            'metadata' => $metadata,
            'success_url' => $returnUrl,
            'cancel_url' => $returnUrl
        ]);
        
        return $response->withJSON(['session_id' => $session]);
    }

    public function webhook($request, $response, $args) {
        $storeSettings = Setting::where('category','store')->get()->keyBy('setting')->toArray();

        $payload = @file_get_contents('php://input');
        $event = null;

        try {
            $event = \Stripe\Webhook::constructEvent($payload, $request->getHeader('Stripe-Signature')[0], $storeSettings['stripe_webhook_signing_secret']['value']);
        } catch(\UnexpectedValueException $e) {
            return $response->withStatus(400);
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            return Store::logFailedPayment('Invalid signature', 'stripe', null, $response);
        }

        if ($event->type == 'checkout.session.completed') {
            $checkoutSession = $event->data->object;

            if (StorePayment::where('processor_id', $checkoutSession->payment_intent)->exists())
                return Store::logFailedPayment('Duplicate payment_intent', 'stripe', $checkoutSession->payment_intent, $response);

            Store::handlePayment([
                'processor' => 'stripe',
                'processor_id' => $checkoutSession->payment_intent,
                'total' => (int) $checkoutSession->metadata->total,
                'currency' => strtoupper($checkoutSession->currency),
                'type' => $checkoutSession->metadata->type,
                'quantity' => $checkoutSession->metadata->quantity ?? 1,
                'package_id' => $checkoutSession->metadata->package_id ?? null,
                'user_id' => $checkoutSession->metadata->user_id
            ], $response);
        }

        return $response->withStatus(200);
    }
}
