<?php
namespace PaypalIntegration\Controllers;

use Ahc\Jwt\JWTException;
use overint\PaypalIPN;
use App\Helpers;
use App\Controllers\Controller;
use App\Models\Setting;
use App\Models\StorePayment;
use App\Models\User;
use App\Store;
use Exception;

class PaypalController extends Controller
{
    public function ipn($request, $response, $args) {
        $settings = Setting::getByCategory('store');
        $params = $request->getParams();

        $ipn = new PaypalIPN();

        if ($settings['paypal_sandbox_mode'])
            $ipn->useSandbox();

        try {
            if (!$ipn->verifyIPN()) {
                if (!$settings['paypal_sandbox_mode'] && isset($params['test_ipn'])) {
                    return Store::logFailedPayment('Received from sandbox while live', 'paypal', $params['txn_id'], $response);
                } else {
                    return Store::logFailedPayment('Unauthorized', 'paypal', $params['txn_id'], $response);
                }
            }
        } catch (Exception $e) {
            return Store::logFailedPayment($e->getMessage(), 'paypal', $params['txn_id'], $response);
        }

        $receiverEmail = strtolower($params['receiver_email']);
        if ($receiverEmail !== strtolower($settings['paypal_email'])) {
            return Store::logFailedPayment("Receiver email mismatch ({$receiverEmail})", 'paypal', $params['txn_id'], $response);
        }
        
        if (StorePayment::where('processor_id', $params['txn_id'])->exists()) {
            return Store::logFailedPayment('Duplicate txn_id', 'paypal', $params['txn_id'], $response);
        }
        
        if ($params['payment_status'] != 'Completed' && $params['payment_status'] != 'Reversed') {
            return Store::logFailedPayment("Payment not completed (status: {$params['payment_status']})", 'paypal', $params['txn_id'], $response);
        }
        
        if ($params['mc_currency'] != $settings['currency']) {
            return Store::logFailedPayment("Invalid currency ({$params['mc_currency']})", 'paypal', $params['txn_id'], $response);
        }

        $total = (int) round($params['mc_gross'] * 100);

        if ($params['payment_status'] === 'Completed' && $params['txn_type'] !== 'adjustment') {
            $id = $params['txn_id'];

            try {
                $payload = Helpers::jwt()->decode($params['custom']);
            } catch (JWTException $e) {
                return Store::logFailedPayment('Failed to decode JWT', 'paypal', $id, $response);
            }

            if ($total < (int) $payload['total']) {
                return Store::logFailedPayment("Paid amount ($total) less than required by order contents ({$payload['total']})", 'paypal', $id, $response);
            }

            Store::handlePayment([
                'processor' => 'paypal',
                'processor_id' => $id,
                'total' => $total,
                'currency' => $params['mc_currency'],
                'type' => $payload['type'],
                'quantity' => $payload['quantity'],
                'package_id' => $payload['package_id'] ?? null,
                'user_id' => $payload['user_id']
            ], $response);
        } elseif ($params['payment_status'] === 'Reversed') {
            $payment = StorePayment::where([
                'processor' => 'paypal',
                'processor_id' => $params['parent_txn_id']
            ])->first();
            $user = $payment->user;
            $user->ban('global', 0, 'Payment reversal');
            User::find('76561197960275975')->notifications()->create([
                'type' => 'payment_reversal',
                'json' => [
                    'txn_id' => $params['txn_id'],
                    'parent_txn_id' => $params['parent_txn_id'],
                    'user_id' => $user->id
                ]
            ]);
        }

        return $response->withStatus(200);
    }
}
