<?php
namespace App\Traits;

use App\Models\BaseModel;
use App\Models\Webhook;

trait Webhookable
{
    /**
     * The attributes that are ignored for triggering webhooks. Always includes 'updated_at'.
     */
    protected $webhookIgnored = [];

    private static $webhookTriggerEvents = ['created', 'updated', 'saved', 'deleted'];

    protected static function booted() {
        if (self::class instanceof Webhook) return;

        foreach (self::$webhookTriggerEvents as $event) {
            static::$event(function (BaseModel $model) use ($event) {
                $modified = $model->isDirty(array_filter(array_keys($model->attributes), function ($attribute) use ($model) {
                    if (!in_array($attribute, array_merge(['updated_at'], $model->webhookIgnored))) return $attribute;
                }));
                if (!$modified) return;

                $path = explode('\\', get_class($model));
                $modelClass = end($path);

                $webhooks = Webhook::where([
                    'model' => $modelClass,
                    'event' => $event,
                    'enabled' => true
                ])->get();

                foreach ($webhooks as $webhook) {
                    if ($webhook->condition && !self::evaluateCondition($webhook->condition, $model)) return;

                    $url = self::interpolateExpressions($webhook->url, $model);
                    $body = self::interpolateExpressions($webhook->body, $model);

                    $headers = $webhook->headers;
                    if (strpos(strtolower(implode('~', $headers)), 'content-type') === false)
                        $headers = array_merge(['Content-Type: application/json'], $headers);

                    $ch = curl_init();
                    curl_setopt($ch, CURLOPT_URL, $url);
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_TIMEOUT, 1);
                    curl_exec($ch);
                    curl_close($ch);
                }
            });
        }
    }

    private static function interpolateExpressions(string $string, BaseModel $model) {
        preg_match_all('({{(.*?)}})', $string, $expressions);
        foreach($expressions[1] as $expression) {
            $parts = explode('|', $expression);

            // Resolve attribute.
            $attribute = $model;
            foreach (array_map('trim', explode('.', $parts[0])) as $subAttribute) {
                $subAttributeName = str_replace('_original', '', $subAttribute);
                $attribute = ($subAttribute === $subAttributeName) ? $attribute->$subAttribute : $attribute->getOriginal($subAttributeName);
            }

            // Apply filters.
            foreach (array_slice($parts, 1) as $filter) {
                $filter = array_map('trim', explode('(', $filter));

                if (array_key_exists(1, $filter))
                    preg_match('/(.*?)\)/', $filter[1], $arg);

                if ($filter[0] === 'truncate')
                    $attribute = substr($attribute, 0, $arg[1] ?? 100);
                else if ($filter[0] === 'lower')
                    $attribute = strtolower($attribute);
                else if ($filter[0] === 'upper')
                    $attribute = strtoupper($attribute);
                else if ($filter[0] === 'capitalize')
                    $attribute = ucfirst($attribute);
            }

            $replacement = addslashes($attribute);
            $string = str_replace("{{{$expression}}}", $replacement, $string);
        }

        return $string;
    }

    private static function evaluateCondition(string $condition, BaseModel $model) {
        $expressions = explode('&&', preg_replace('/\s/', '', self::interpolateExpressions($condition, $model)));
        foreach($expressions as $expression) {
            preg_match_all('/(.*)?(==|!=)(.*)?/', $expression, $parts);
            $left = $parts[1][0];
            $operator = $parts[2][0];
            $right = $parts[3][0];
            if ($operator === '==') {
                if ($left !== $right) return false;
            } else if ($operator === '!=') {
                if ($left === $right) return false;
            }
        }
        return true;
    }
}
