<?php
namespace App\Controllers;

use Illuminate\Database\Capsule\Manager as DB;
use Slim\Routing\RouteContext;
use App\Models\StoreCredit;
use App\Models\StorePackage;
use App\Models\StorePackagePurchase;
use App\Models\StorePayment;
use App\Models\StoreFailedPaymentLog;
use App\Models\Server;
use App\Models\Setting;
use App\Models\User;
use App\Helpers;
use App\Store;

class StoreController extends Controller
{
    public function getCredits($request, $response) {
        $monthly_revenue = StorePayment::select([DB::raw('SUM(total)')])->whereRaw('MONTH(timestamp) = MONTH(CURDATE())')->whereRaw('YEAR(timestamp) = YEAR(CURDATE())')->first()['SUM(total)'];
        $this->view->getEnvironment()->addGlobal('monthly_revenue', $monthly_revenue);
        $topPurchases = StorePayment::select(['user_id', DB::raw('SUM(total)')])->groupBy('user_id')->orderByRaw('SUM(total) DESC')->take(5)->get();
        $this->view->getEnvironment()->addGlobal('top_purchases', $topPurchases);
        $recentPurchases = StorePayment::orderBy('timestamp', 'DESC')->take(5)->get();
        $this->view->getEnvironment()->addGlobal('recent_purchases', $recentPurchases);
        $this->view->getEnvironment()->addGlobal('payment_processors', $this->store);
        return $this->view->render($response, 'store/credits.twig');
    }

    public function getPurchases($request, $response, $args) {
        $this->view->getEnvironment()->addGlobal('user', isset($args['id']) ? User::find($args['id']) : $this->auth->user());
        if ($this->auth->isAdmin())
            $this->view->getEnvironment()->addGlobal('servers', Server::whereHas('packages')->with('packages')->orderBy('order', 'ASC')->get());
        $this->view->getEnvironment()->addGlobal('payment_processors', $this->store);
        return $this->view->render($response, 'store/purchases.twig');
    }

    public function servers($request, $response, $args) {
        $servers = Server::orderBy('order','asc')->has('packages')->get();
        if (count($servers) === 1) {
            return $response->withRedirect(RouteContext::fromRequest($request)->getRouteParser()->urlFor('store_packages', ['serverid'=>$servers->first()->id]));
        }
        $this->view->getEnvironment()->addGlobal('servers', $servers);
        return $this->view->render($response, 'store/serverpicker.twig');
    }

    public function packages($request, $response, $args) {
        $storePackages = StorePackage::where('server',$args['serverid'])->orderBy('order','ASC')->get()->append('active_purchases')->append('purchase_count');
        $storePackages = Helpers::whitelist_keys($storePackages->toArray(), ['*'=>['id','name','cost_credits','cost_currency','valid_for','purchase_limit','image','description','short_description','active_purchases','purchase_count']]);
        $this->view->getEnvironment()->addGlobal('storepackages', $storePackages);
        $this->view->getEnvironment()->addGlobal('payment_processors', $this->store);
        return $this->view->render($response, 'store/packages.twig');
    }

    public function getPackagePurchases($request, $response) {
        $params = $request->getParams();

        $data = StorePackagePurchase::orderBy('purchase_timestamp', 'desc')->with(['user', 'storePayment']);
        if (!empty($params['payment_id']))
            $data = $data->where('payment_id', $params['payment_id']);
        $data = $data->paginate($params['perPage'], ['*'], 'page', $params['targetPage']);
        $data->getCollection()->map->user->append('avatar');

        return $response->withJSON($data);
    }

    public function getUserPackagePurchases($request, $response, $args) {
        $params = $request->getParams();
        $data = User::find($args['user_id'])->storePackagePurchases()->orderBy('purchase_timestamp', 'DESC')->paginate($params['perPage'], ['*'], 'page', $params['targetPage']);
        return $response->withJSON($data);
    }

    public function getPayments($request, $response) {
        $params = $request->getParams();

        $data = StorePayment::orderBy('timestamp', 'desc')->with(['user']);
        if (!empty($params['processor_id']))
            $data = $data->where('processor_id', 'like', "%{$params['processor_id']}%");
        $data = $data->paginate($params['perPage'], ['*'], 'page', $params['targetPage']);
        $data->getCollection()->map->user->append('avatar');

        return $response->withJSON($data);
    }

    public function getUserPayments($request, $response, $args) {
        $params = $request->getParams();
        $data = User::find($args['user_id'])->storePayments()->orderBy('timestamp', 'DESC')->paginate($params['perPage'], ['*'], 'page', $params['targetPage']);
        return $response->withJSON($data);
    }

    public function getFailedPaymentLogs($request, $response) {
        $params = $request->getParams();

        $data = StoreFailedPaymentLog::orderBy('timestamp', 'desc');
        if (!empty($params['filter_id'])) {
            $data = $data->where('processor_id', 'like', "%{$params['filter_id']}%");
        }
        $data = $data->paginate($params['perPage'], ['*'], 'page', $params['targetPage']);

        return $response->withJSON($data);
    }

    public function purchasePackage($request, $response, $args) {
        $package = StorePackage::find($args['id']);

        if ($package != null) {
            $settings = Setting::getByCategory('store');
            $user = $this->container->auth->user();

            if (!($settings['credits_enabled'] || $package->cost_currency <= 0 ))
                throw new \Slim\Exception\HttpForbiddenException($request);

            if (Store::purchaseLimitReached($user, $package))
                return $response->withJSON(['status' => 'purchase_limit_reached']);

            if ($package->cost_credits <= ($user->credits ? $user->credits->credits : 0)) {
                DB::transaction(function() use ($user, $package) {
                    if($package->cost_credits > 0)
                        $user->credits->update(['credits'=>DB::raw("credits-'{$package->cost_credits}'")]);

                    Store::grantPackage($package, $user, null);
                });
            } else {
                return $response->withJSON(['status' => 'insufficient_funds']);
            }
        } else {
            return $response->withStatus(404);
        }

        return $response->withJSON(['status' => 'success']);
    }

    public function postSign($request, $response, $args) {
        $params = $request->getParams();
        $data = [
            'user_id' => $this->auth->id(),
            'type' => $params['type'],
            'quantity' => $params['quantity'] ?? 1,
            'total' => $request->getAttribute('total')
        ];
        if ($params['type'] == 'package') {
            $data['package_id'] = $request->getAttribute('package')->id;
        }
        $jwt = Helpers::jwt()->encode($data);
        return $response->withJSON(['token' => $jwt]);
    }

    public function updateCredits($request, $response, $args) {
        StoreCredit::firstOrCreate(['user_id' => $args['id']])->update(['credits' => $request->getParam('amount')]);
        return $response->withJSON(['status' => 'success']);
    }

    public function storePackage($request, $response, $args) {
        Store::grantPackage(StorePackage::find($request->getParam('id')), User::find($args['id']), null);
        return $response->withJSON(['status' => 'success']);
    }
}
