<?php
namespace App\Controllers;

use App\Helpers;
use App\Models\User;
use App\Models\Role;
use App\Models\Feature;
use App\Models\Setting;
use App\Models\Server;
use App\Models\StaffMember;
use App\Models\LoadingScreen;

class PagesController extends Controller
{
    public function moduleAsset($request, $response, $args) {
        $file = str_replace('..', '', $request->getParam('f'));
        header_remove('Cache-Control'); header_remove('Pragma'); $cache_sec = 86400; $exp = gmdate("D, d M Y H:i:s", time() + $cache_sec) . " GMT";
        $response->getBody()->write(file_get_contents(__DIR__ . "/../../modules/{$args['identifier']}/public/$file"));
        $contentType = 'application/javascript'; if (strpos($file, '.css')) { $contentType = 'text/css'; }
        return $response->withHeader('Cache-Control', "public, max-age=$cache_sec")->withHeader('Expires', $exp)->withHeader('Content-Type', $contentType);
    }

    public function landing($request, $response) {
        if (Setting::find('display_community')->value??false) {
            $community = $this->cache->get('community');
            if (empty($community)) {
                $community = [
                    'monthly_players' => User::whereNotNull('last_played')->whereRaw('MONTH(last_played) = MONTH(NOW())')->count(),
                    'total_players' => User::whereNotNull('last_played')->count(),
                    'registered_users' => User::whereNotNull('last_online')->count()
                ];
                $this->cache->put('community', $community, 300);
            }
            $this->view->getEnvironment()->addGlobal('community', $community);
        }
        $this->view->getEnvironment()->addGlobal('servers', Server::get()->keyBy('id')->map->append('imageURL')->toJson());
        $this->view->getEnvironment()->addGlobal('features', Feature::orderBy('order','asc')->get()->toArray());
        $this->view->getEnvironment()->addGlobal('staffmembers', StaffMember::orderBy('order','asc')->get());
        return $this->view->render($response, 'landing.twig');
    }

    public function bans($request, $response) {
        if ($this->auth->canBan()) {
            $this->view->getEnvironment()->addGlobal('servers', Server::orderBy('order','asc')->get());
        }
        return $this->view->render($response, 'bans.twig');
    }

    public function users($request, $response) {
        $this->view->getEnvironment()->addGlobal('roles', Role::orderBy('order', 'ASC')->get());
        return $this->view->render($response, 'users.twig');
    }

    public function user($request, $response, $args) {
        global $settings;

        if (isset($args['id']))
            $arr = ['id' => $args['id']];
        else if (isset($args['steamid']))
            $arr = ['steamid' => $args['steamid']];

        $model = User::class;
        if (array_key_exists('forums', $this->modules))
            $model = \Forums\Models\ForumUser::class;

        $user = $model::where($arr)->first();

        if (!isset($user))
            throw new \Slim\Exception\HttpNotFoundException($request);

        if ($this->auth->isAdmin())
            $this->view->getEnvironment()->addGlobal('roles', Role::orderBy('order', 'ASC')->get());

        $this->view->getEnvironment()->addGlobal('user', $user);

        if ($user->id == $this->auth->id()) {
            if (User::whereNotNull('minecraft_uuid')->exists()) {
                $this->view->getEnvironment()->addGlobal('minecraft_auth', true);
                if ($user->minecraft_uuid == null)
                    $this->view->getEnvironment()->addGlobal('link_command_minecraft', '/link ' . Helpers::jwt()->encode(['auth_user_id' => $this->auth->id()]));
            }
            $this->view->getEnvironment()->addGlobal('discord_oauth', !empty($settings['discord_api']['client_secret']));
        }

        return $this->view->render($response, 'profile.twig');
    }

    public function tos($request, $response, $args) {
        return $this->view->render($response, 'legal/tos.twig');
    }

    public function privacy($request, $response, $args) {
        return $this->view->render($response, 'legal/privacy.twig');
    }

    public function impressum($request, $response, $args) {
        return $this->view->render($response, 'legal/impressum.twig');
    }

    public function loading($request, $response, $args) {
        $steamid = $request->getParam('steamid');
        if (empty($steamid) || $steamid == '%s') {
            $steamid = '76561197960275975';
        }
        if(!preg_match('/7656119[0-9]{10}/i', $steamid)) {
            die('Invalid SteamID');
        }

        $loadingScreen = LoadingScreen::find($args['id']);
        $user = User::firstOrCreate(['steamid' => $steamid]);

        if ($loadingScreen->db_username != null && $loadingScreen->db_password != null) {
            $db = [
                'driver' => 'mysql',
                'host' => $loadingScreen->db_host ?? '127.0.0.1',
                'port' => $loadingScreen->db_port ?? 3306,
                'username' => $loadingScreen->db_username,
                'password' => $loadingScreen->db_password
            ];

            $dbData = [];

            if ($loadingScreen->db_darkrp_database) {
                $this->capsule->addConnection(array_merge($db, ['database' => $loadingScreen->db_darkrp_database]), 'darkrp');
                $dbData['darkrp'] = $user->darkRpPlayer;
            }

            if ($loadingScreen->db_pointshop_database) {
                $this->capsule->addConnection(array_merge($db, ['database' => $loadingScreen->db_pointshop_database]), 'pointshop');
                $dbData['pointshop'] = $user->pointshopPlayer;
            }

            if ($loadingScreen->db_pointshop2_database) {
                $this->capsule->addConnection(array_merge($db, ['database' => $loadingScreen->db_pointshop2_database]), 'pointshop2');
                $dbData['pointshop2'] = $user->pointshop2_wallet;
            }

            $this->view->getEnvironment()->addGlobal('db_data', $dbData);
        }

        $this->view->getEnvironment()->addGlobal('user', $user);
        $this->view->getEnvironment()->addGlobal('loadingscreen', $loadingScreen->toArray());
        return $this->view->render($response, 'loading.twig');
    }

    public function adminPackagesServerPicker($request, $response, $args) {
        $this->view->getEnvironment()->addGlobal('servers', Server::orderBy('order','asc')->get());
        return $this->view->render($response, 'store/serverpicker.twig');
    }

    public function adminPackages($request, $response, $args) {
        $server = Server::find($args['server']);
        $this->view->getEnvironment()->addGlobal('server', ['id'=>$args['server'],'name'=>$server->name,'game'=>$server->game]);
        $this->view->getEnvironment()->addGlobal('roles', Role::orderBy('order','asc')->disableCache()->whereDoesntHave('excludedServers', function ($q) use ($args) {
            $q->where('server', $args['server']);
        })->get());
        return $this->view->render($response, 'admin/packages.twig');
    }

    public function notifications($request, $response, $args) {
        return $this->view->render($response, 'notifications.twig');
    }

    public function error($request, $response, $args) {
        $this->view->getEnvironment()->addGlobal('errorCode', $args['code']);
        return $this->view->render($response, 'error.twig');
    }
}
