<?php

namespace App\Controllers;

use Illuminate\Database\Capsule\Manager as DB;
use App\Models\Ban;
use App\Models\Setting;

class BanController extends ApiResourceController
{
    public function __construct($container) {
        parent::__construct($container);
        $this->model = Ban::class;
        $this->columns = ['id', 'scope', 'server_id', 'offender_user_id', 'created', 'expires', 'reason', 'admin_user_id'];
    }

    public function index($request, $response, $args) {
        if ((Setting::find('ban_list_visibility')->value ?? null) != 'public' && !$this->auth->isAdmin())
            throw new \Slim\Exception\HttpForbiddenException($request);

        $params = $request->getParams();

        if (isset($params['offender_user_id'])) {
            $bans = Ban::where('offender_user_id', '=', $params['offender_user_id'])->orderBy('id', 'DESC');
        } else if (!empty($params['filter_user_name'])) {
            $bans = Ban::whereHas('offender', function($query) use ($params) {
                    $query->where('name', 'like', "%{$params['filter_user_name']}%");
                })
                ->orWhereHas('admin', function ($query) use ($params) {
                    $query->where('name', 'like', "%{$params['filter_user_name']}%");
                })
                ->orderBy('id', 'DESC');
        } else {
            $bans = Ban::orderBy('id', 'DESC');
        }

        $bans = $bans->with(['offender', 'admin', 'server'])->paginate($params['perPage'], ['*'], 'page', $params['targetPage']);

        $now = new \DateTime(DB::select(DB::raw('SELECT NOW() AS now'))[0]->now);

        $bans->getCollection()->map(function($ban) use ($now) {
            if ($ban->offender) $ban->offender->append('avatar');
            if ($ban->admin) $ban->admin->append('avatar');
            $ban->expired = (new \DateTime($ban->expires) < $now);
        });

        return $response->withJSON($bans);
    }

    public function store($request, $response, $args) {
        $params = $request->getParams();
        $ban = Ban::create([
            'scope' => $params['scope'] ?? 'global',
            'server_id' => !empty($params['server_id']) ? $params['server_id'] : null,
            'offender_user_id' => $params['offender_user_id'],
            'created' => DB::raw('NOW()'),
            'expires' => $params['length'] != 0 ? DB::raw('DATE_ADD(NOW(), INTERVAL ' . $params['length'] . ' MINUTE)') : null,
            'reason' => $params['reason'],
            'admin_user_id' => $this->container->auth->id()
        ]);
        $ban->flushCache();
        return $response->withJSON(['status' => 'success']);
    }

    public function unban($request, $response, $args) {
        Ban::where('id', $args['id'])->update(['expires' => DB::raw('NOW()')]);
        return $response->withStatus(204);
    }
}
