<?php
namespace App\Controllers;

use App\Controllers\Controller;
use Slim\Routing\RouteContext;
use Illuminate\Database\Capsule\Manager as DB;
use App\Models\User;
use App\Models\Setting;;
use Hybridauth;

class AuthController extends Controller
{
    public function show($request, $response) {
        global $settings;

        $params = $request->getParams();
        $routeParser = RouteContext::fromRequest($request)->getRouteParser();

        if ((Setting::find('primary_auth_provider')->value ?? 'steam') !== 'any')
            return $response->withRedirect($routeParser->urlFor('auth_provider', ['provider' => 'steam'], ['redirect' => $params['redirect']]));

        $this->view->getEnvironment()->addGlobal('discord_oauth', !empty($settings['discord_api']['client_secret']));
        $this->view->getEnvironment()->addGlobal('redirect', $params['redirect'] ?? null);
        return $this->view->render($response, 'auth.twig');
    }

    public function authenticate($request, $response, $args) {
        global $settings;

        $params = $request->getParams();
        $routeParser = RouteContext::fromRequest($request)->getRouteParser();

        if (session_status() !== PHP_SESSION_ACTIVE) {
            session_start();
        } else {
            session_regenerate_id();
        }

        if (isset($params['redirect']))
            $this->auth->setRedirectUrl($params['redirect']);

        if ((Setting::find('primary_auth_provider')->value ?? 'steam') !== 'any' && $args['provider'] !== 'steam')
            return $response->withRedirect($routeParser->urlFor('auth_provider', ['provider' => 'steam']));

        $uri = $request->getUri();
        $fwdS = (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https' && $uri->getScheme() == 'http') ? 's': '';
        $config = ['callback' => "{$uri->getScheme()}{$fwdS}://{$uri->getHost()}{$uri->getPath()}"];

        if ($args['provider'] === 'steam') {
            $adapter = new Hybridauth\Provider\Steam($config);
        } else if ($args['provider'] === 'discord') {
            $config['keys'] = [
                'key' => $settings['discord_api']['client_id'],
                'secret' => $settings['discord_api']['client_secret']
            ];

            $addToGuild = isset($settings['discord_api']['bot_token']) && isset($settings['discord_api']['guild_id']);
            if ($addToGuild) {
                $config['scope'] = 'identify email guilds.join';
            }

            $adapter = new Hybridauth\Provider\Discord($config);
        } else {
            throw new \Slim\Exception\HttpNotFoundException($request);
        }

        $adapter->authenticate();

        $userProfile = $adapter->getUserProfile();

        if ($userProfile->identifier == null)
            return $response->withStatus(503);

        $name = $userProfile->displayName;
        if ($args['provider'] === 'discord')
            $name = substr($name, 0, strpos($name, '#'));

        $idColumn = "{$args['provider']}id";
        if ($this->auth->check()) {
            $user = $this->auth->user();
            if ($user->{$idColumn} !== $userProfile->identifier) {
                if (User::where($idColumn, $userProfile->identifier)->exists())
                    throw new \Slim\Exception\HttpForbiddenException($request);
                $user->update([$idColumn => $userProfile->identifier]);
            }
        } else {
            $user = User::updateOrCreate([$idColumn => $userProfile->identifier], ['name' => $name, 'last_online' => DB::raw('NOW()')]);
            $this->auth->setId($user->id);
        }

        if (isset($userProfile->photoURL))
            $user->avatar = $userProfile->photoURL;

        if (isset($addToGuild)) {
            try {
                $data = json_encode([
                    'access_token' => $adapter->getAccessToken()['access_token']
                ]);
                $ch = curl_init();
                curl_setopt_array($ch, [
                    CURLOPT_HTTPHEADER => ["Authorization: Bot {$settings['discord_api']['bot_token']}", 'Content-Type: application/json', 'Content-Length: ' . strlen($data)],
                    CURLOPT_URL => "https://discordapp.com/api/guilds/{$settings['discord_api']['guild_id']}/members/{$userProfile->identifier}",
                    CURLOPT_POSTFIELDS => $data,
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_CUSTOMREQUEST => "PUT",
                    CURLOPT_TIMEOUT => 3
                ]);
                curl_exec($ch);
                curl_close($ch);
            } catch (\Exception $e) { }
        }

        return $response->withRedirect($this->auth->consumeRedirectUrl() ?? RouteContext::fromRequest($request)->getRouteParser()->urlFor('user', ['id' => $this->auth->id()]));
    }

    public function logout($request, $response) {
        session_destroy();
        return $response->withRedirect($request->getParam('redirect') ?? RouteContext::fromRequest($request)->getRouteParser()->urlFor('landing'));
    }

    public function impersonate($request, $response, $args) {
        if (!$this->container->auth->isOwner())
            throw new \Slim\Exception\HttpForbiddenException($request);

        $this->auth->impersonateId($args['id']);
        return $response->withStatus(204);
    }

    public function unimpersonate($request, $response) {
        $this->auth->unimpersonate();
        return $response->withRedirect($request->getParam('redirect') ?? RouteContext::fromRequest($request)->getRouteParser()->urlFor('user', ['id' => $this->auth->id()]));
    }
}
