<?php
namespace App\Models;

use Illuminate\Database\Capsule\Manager as DB;

class User extends BaseModel
{
    protected $primaryKey = 'id';
    public $incrementing = true;
    public $timestamps = false;
    protected $guarded = [];
    protected $appends = ['status'];
    protected $casts = [
        'steamid' => 'string'
    ];
    protected $dispatchesEvents = [
        'saving' => \App\Events\UserSaving::class,
        'saved' => \App\Events\UserSaved::class
    ];
    public $ignoreSavingEvent;
    protected $webhookIgnored = ['last_online', 'last_played'];

    public function group() {
        return $this->hasOneThrough(Role::class, 'App\Models\UserRole', 'user_id', 'rid', 'id', 'rid')->orderBy('order');
    }

    public function userRoles() {
        return $this->hasMany(UserRole::class, 'user_id', 'id');
    }

    public function roles() {
        return $this->hasManyThrough(Role::class, 'App\Models\UserRole', 'user_id', 'rid', 'id', 'rid')->orderBy('order');
    }

    public function userRevokedRoles() {
        return $this->hasMany(UserRevokedRole::class, 'user_id', 'id');
    }

    public function revokedRoles() {
        return $this->hasManyThrough(Role::class, 'App\Models\UserRevokedRole', 'user_id', 'rid', 'id', 'rid');
    }

    public function bans() {
        return $this->hasMany(Ban::class, 'offender_user_id', 'id');
    }

    public function issuedBans() {
        return $this->hasMany(Ban::class, 'admin_user_id', 'id');
    }

    public function storePackagePurchases() {
        return $this->hasMany(StorePackagePurchase::class);
    }

    public function storePayments() {
        return $this->hasMany(StorePayment::class);
    }

    public function darkRpPlayer() {
        return $this->hasOne('App\Models\DarkRpPlayer', 'uid', 'steamid');
    }

    public function pointshopPlayer() {
        return $this->hasOne('App\Models\PointshopPlayer', 'uniqueid', 'uniqueid');
    }

    public function addRole($rid) {
        $where = ['user_id' => $this->id, 'rid' => $rid];
        return UserRole::firstOrCreate($where) && UserRevokedRole::where($where)->delete();
    }

    public function removeRoles($roles) {
        $userRoles = UserRole::where('user_id', $this->id)->whereIn('rid', $roles->pluck('rid'));
        foreach ($userRoles->get() as $role) {
            UserRevokedRole::firstOrCreate(['user_id' => $this->id, 'rid' => $role->rid]);
        }
        return $userRoles->delete();
    }

    public function removeRole($rid) {
        $where = ['user_id' => $this->id, 'rid' => $rid];
        return UserRole::where($where)->delete() && UserRevokedRole::firstOrCreate($where);
    }

    public function isBannedFromWeb() {
        return Ban::where('offender_user_id', $this->id)->whereIn('scope', ['global', 'web'])->where(function($query) {
            $query->whereRaw('expires > NOW()')->orWhereNull('expires');
        })->exists();
    }

    public function hasPermission($name) {
        $permission = RolePermission::where('name', $name)->where('type', 'allow');

        if ($this->isBannedFromWeb()) {
            return $permission->where('subject', 'banned')->exists();
        }

        return $permission->where(function($query) {
            $query->whereIn('rid', $this->roles->pluck('rid'))->orWhere('subject', 'everyone');
        })->exists();
    }

    public function credits() {
        return $this->hasOne('App\Models\StoreCredit');
    }

    public function notifications() {
        return $this->hasMany(Notification::class, 'user_id')->orderBy('nid','DESC');
    }

    public function unread_notifications() {
        return $this->notifications()->where('read',NULL);
    }

    public function ban($scope = 'web', $length = 0, $reason = null) {
        return Ban::create([
            'scope' => $scope,
            'offender_user_id' => $this->id,
            'expires' => $length > 0? DB::raw('DATE_ADD(NOW(), INTERVAL '. $length .' MINUTE)'): null,
            'reason' => $reason
        ]);
    }

    public function getSteamAttribute() {
        if (empty($this->steamid)) return null;
        self::cacheSteamProfiles([$this]);
        global $container;
        return (object) $container->cache->get("steamProfile_{$this->steamid}");
    }

    public function getSteamid32Attribute() {
        return 'STEAM_0:' . ($this->steamid % 2) . ':' . ($this->steamid - (76561197960265728 + ($this->steamid % 2)))/2;
    }

    public function getUniqueidAttribute() {
        return crc32("gm_{$this->steamid32}_gm");
    }

    public function setAvatarAttribute($url) {
        global $container;
        $container->cache->put("user_{$this->id}_avatar", $url);
    }

    public function getAvatarAttribute() {
        global $container;
        $avatar = $container->cache->get("user_{$this->id}_avatar");
        if (!empty($avatar))
            return $avatar;
        if (!empty($this->steamid) && property_exists($this->steam, 'avatarfull'))
            return $this->steam->avatarfull;
        if (!empty($this->minecraft_uuid))
            return "https://crafatar.com/avatars/{$this->minecraft_uuid}?size=184&overlay=true";
        return '/img/avatar.svg';
    }

    public function getNameAttribute($value) {
        if (empty($value))
            return $this->steam->personaname ?? 'Unknown';
        return $value;
    }

    public function getStatusAttribute() {
        $lastOnline = $this->last_online == null ? -1 : strtotime($this->last_online.'Z');
        if ($lastOnline > time()-(60*3)) {
            return 'online';
        } else if ($lastOnline > time()-(60*15)) {
            return 'away';
        } else {
            return 'offline';
        }
    }

    public function unredeemedPurchases($serverID) {
        return $this->storePackagePurchases()->unredeemed()->valid()->whereHas('storePackage', function ($q) use ($serverID) {
            $q->where('server', $serverID);
        })->with('storePackage');
    }

    public function expiringPurchases($serverID) {
        return $this->storePackagePurchases()->expiring()->whereHas('storePackage', function ($q) use ($serverID) {
            $q->where('server', $serverID);
        })->with('storePackage');
    }

    public function isBannedOnServer($serverID, $banScope = null) {
        if(!isset($banScope) || isset($banScope) && $banScope != 'none') {
            return Ban::where('offender_user_id', $this->id)->where(function ($q) use ($serverID, $banScope) {
                if(!isset($banScope) || $banScope == 'server') {
                    $q->where('server_id', $serverID)->where('scope', 'server')->orWhere('scope', 'global');
                }
            })->where(function($q) {
                $q->whereRaw('expires > NOW()')->orWhereNull('expires');
            })->exists();
        }
        return false;
    }

    public function permaWeapons($serverID) {
        $closure = function ($q) use ($serverID) {
            $q->where('server', $serverID);
            $q->whereNotNull('perma_weapons');
        };
        $packagePurchases = $this->storePackagePurchases()->valid()->whereHas('storePackage', $closure)->with(['storePackage' => $closure])->get();
        return $packagePurchases->pluck('storePackage')->pluck('perma_weapons')->flatten()->unique();
    }

    public function getPointshop2WalletAttribute() {
        return DB::connection('pointshop2')->table('libk_player')->join('ps2_wallet', 'libk_player.id', '=', 'ps2_wallet.ownerId')->select('ps2_wallet.points', 'ps2_wallet.premiumPoints')->where('libk_player.steam64', '=', $this->steamid)->first();
    }

    static function cacheSteamProfiles($users) {
        global $container;
        $steamIds = '';

        $users = collect($users)->filter(function($user) {
            return !empty($user->steamid);
        });

        foreach ($users as $user)
        	if (!$container->cache->has("steamProfile_{$user->steamid}"))
                $steamIds .= $user->steamid . ',';

        if (!empty($steamIds)) {
            $steamApiKey = $container->settings['steam_api_key'];
            $response = json_decode(file_get_contents("https://api.steampowered.com/ISteamUser/GetPlayerSummaries/v0002/?key={$steamApiKey}&steamids={$steamIds}"), true);
            $players = collect($response['response']['players'])->keyBy('steamid');
            foreach ($users as $user)
                $container->cache->put("steamProfile_{$user->steamid}", $players->get($user->steamid) ?? false, 604800);
        }
    }

    /**
     * Cache data that isn't obtainable from the database for multiple users at once.
     */
    static function cache($users) {
        self::cacheSteamProfiles($users);
    }

    /**
     * Update foreign keys on associated Ban, UserRole & UserRevokedRole records to point to a target user and delete this user.
     */
    function merge(User $targetUser) {
        if ($targetUser->minecraft_uuid !== null)
            return false;

        return DB::transaction(function () use ($targetUser) {
            $this->bans()->update(['offender_user_id' => $targetUser->id]);
            $this->issuedBans()->update(['admin_user_id' => $targetUser->id]);
            $this->userRoles()->whereNotIn('rid', $targetUser->userRoles()->pluck('rid'))->update(['user_id' => $targetUser->id]);
            $this->userRevokedRoles()->whereNotIn('rid', $targetUser->userRoles()->pluck('rid')->merge($targetUser->userRevokedRoles()->pluck('rid')))->update(['user_id' => $targetUser->id]);
            $this->delete();
            $targetUser->update([
                'minecraft_uuid' => $this->minecraft_uuid,
                'created' => min($this->created, $targetUser->created),
                'last_played' => max($this->last_played, $targetUser->last_played)]);
            return true;
        });
    }
}
