<?php
namespace App\Controllers;

use App\Helpers;
use Illuminate\Database\Capsule\Manager as DB;
use App\Models\User;
use App\Models\Role;
use App\Models\Ban;
use App\Models\StorePackagePurchase;

class ServerApiController extends Controller
{
    private $idColumn;

    function getConnectionCheck($request, $response, $args) {
        $server = $request->getAttribute('server');
        return $response->withJSON([
            'status' => 'success',
            'settings' => [
                'ban_log' => $server->ban_log ?? false,
                'role_sync' => [
                    'receive' => in_array($server->role_sync, ['true', 'receive']),
                    'send' => in_array($server->role_sync, ['true', 'send']),
                    'create' => $server->role_sync_create ?? false
                ],
                'poll_interval' => $server->poll_interval ?? 0
            ]
        ]);
    }

    /**
     * Determine ID column used to map players to users based on the request object.
     */
    function determineIdColumn($request) {
        $this->idColumn = 'steamid';
        if ($request->getAttribute('server')->game == 'minecraft')
            $this->idColumn = 'minecraft_uuid';
    }

    /**
     * Return a batch of structured user data for the game server to process.
     */
    function postUsers($request, $response, $args) {
        $this->determineIdColumn($request);
        $userIds = explode(',', $request->getParam("{$this->idColumn}s"));
        $names = explode(',', urldecode($request->getParam('names')));
        $server = $request->getAttribute('server');

        $responseData = ['users' => []];

        foreach ($userIds as $index => $userId) {
            $values = ['last_played' => DB::raw('CURRENT_TIMESTAMP()')];
            if (array_key_exists($index, $names) && !empty($names[$index]))
                $values['name'] = $names[$index];

            $user = User::updateOrCreate([$this->idColumn => $userId], $values);

            $banned = $user->isBannedOnServer($server->id, $server->ban_scope);

            if (!$banned) {
                $expiringPurchases = $user->expiringPurchases($server->id)->with('storePackage.role')->get() ?? [];
                $unredeemedPurchases = $user->unredeemedPurchases($server->id)->with('storePackage.role')->get() ?? [];

                $closure = function (StorePackagePurchase $purchase) {
                    $purchase->storePackage->interpolateCommandExpressions($purchase);
                    $purchase->setHidden(['id', 'package_id', 'payment_id', 'package_name', 'package_cost_credits', 'user_id', 'user']);
                    $purchase->storePackage->setHidden(['id', 'server', 'purchase_limit', 'rid', 'image', 'description', 'short_description', 'order']);
                    if ($purchase->storePackage->role)
                        $purchase->storePackage->role->setHidden(['rid', 'name', 'color', 'icon', 'order']);
                };

                $expiringPurchases->map($closure);
                $unredeemedPurchases->map($closure);

                $expiringPurchases->map(function (StorePackagePurchase $purchase) use ($user) {
                    if ($purchase->storePackage->role)
                        $user->removeRole($purchase->storePackage->role->rid);
                });

                $roleWhere = function ($q) use ($request) {
                    $q->whereDoesntHave('excludedServers', function ($q) use ($request) {
                        $q->where('server', $request->getAttribute('server')->id);
                    });
                };

                $responseData['users'][$userId] = [
                    $this->idColumn => $userId,
                    'banned' => $banned,
                    'roles' => $user->roles()->where($roleWhere)->disableCache()->pluck('ingame_equivalent') ?? null,
                    'revoked_roles' => $user->revokedRoles()->where($roleWhere)->disableCache()->pluck('ingame_equivalent') ?? null,
                    'store_package_purchases' => [
                        'expiring' => $expiringPurchases,
                        'unredeemed' => $unredeemedPurchases
                    ]
                ];

                if ($request->getAttribute('server')->game == 'gmod')
                    $responseData['users'][$userId]['perma_weapons'] = $user->permaWeapons($server->id);
            } else {
                $responseData['users'][$userId] = [$this->idColumn => $userId, 'banned' => $banned];
            }
        }

        return $response->withJSON($responseData);
    }

    /**
     * Flag a batch of users as successfully processed. Called by the game server after processing their data.
     */
    function postUsersProcessed($request, $response, $args) {
        $this->determineIdColumn($request);
        $userIds = explode(',', $request->getParam("{$this->idColumn}s"));
        $serverID = $request->getAttribute('server')->id;

        $users = User::whereIn($this->idColumn, $userIds)->get();
        foreach ($users as $user) {
            $user->unredeemedPurchases($serverID)->where('purchase_timestamp', '<=', $user->last_played)->update(['redeemed' => 1]);
            $user->expiringPurchases($serverID)->where('expiry_timestamp', '<=', $user->last_played)->update(['expired' => 1]);
        }

        return $response->withStatus(200);
    }

    /**
     * Called periodically by the game server to determine which players' data needs to be reprocessed.
     */
    function getUsersPoll($request, $response, $args) {
        $this->determineIdColumn($request);
        $userIds = explode(',', $request->getParam("{$this->idColumn}s"));
        $server = $request->getAttribute('server');
        
        $users = User::whereIn($this->idColumn, $userIds)->where(function($q) use ($server) {
            $q->whereHas('storePackagePurchases', function ($q) use ($server) {
                $q->where(function ($q) {
                    $q->unredeemed()->orWhere(function ($q) {
                        $q->expiring();
                    });
                })->whereHas('storePackage', function ($q) use ($server) {
                    $q->where('server', $server->id);
                });
            });

            if ($server->ban_scope != 'none')
                $q->orWhereHas('bans', function ($q) use ($server) {
                    $q->where(function ($q) {
                        $q->whereRaw('expires > NOW()')->orWhereNull('expires');
                    })->where(function ($q) use ($server) {
                        if ($server->ban_scope == 'server')
                            $q->where('scope', 'server')->where('server_id', $server->id)->orWhere('scope', 'global');
                    });
                });
        })->get();

        return $response->withJSON(['status' => 'success', 'users' => $users->pluck($this->idColumn)]);
    }

    function postUserBan($request, $response, $args) {
        $params = $request->getParams();
        $this->determineIdColumn($request);
        $offender = User::updateOrCreate([$this->idColumn => $args['id']]);

        if (!empty($params["admin_{$this->idColumn}"]))
            $admin = User::where($this->idColumn, $params["admin_{$this->idColumn}"])->first();

        Ban::create([
          'offender_user_id' => $offender->id,
          'scope' => isset($params['global'])? 'global': 'server',
          'server_id' => $request->getAttribute('server')->id,
          'reason' => $params['reason'] ?? null,
          'admin_user_id' => isset($admin) ? $admin->id : null,
          'expires' => $params['expiry_minutes'] > 0? DB::raw('DATE_ADD(NOW(), INTERVAL '. $params['expiry_minutes'] .' MINUTE)'): null
        ])->flushCache();

        return $response->withJSON(['status' => 'success']);
    }

    function deleteUserBan($request, $response, $args) {
        $this->determineIdColumn($request);
        $offender = User::where($this->idColumn, $args['id'])->first();

        if (!$offender) return $response->withJSON(['status' => 'success']);
        
        $success = $offender->bans()->where(function ($q) use ($request) {
            $q->where('server_id', $request->getAttribute('server')->id)->where('scope', 'server')->orWhere('scope', 'global');
        })->where(function($q) {
            $q->whereRaw('expires > NOW()')->orWhereNull('expires');
        })->update([
            'expires' => DB::raw('NOW()')
        ]);

        return $response->withJSON(['status' => $success ? 'success': 'failure']);
    }

    function postUserRole($request, $response, $args) {
        $params = $request->getParams();
        $this->determineIdColumn($request);
        $user = User::where($this->idColumn, $args['id'])->first();
        $server = $request->getAttribute('server');
        $role = Role::where(['ingame_equivalent' => $params['role']])->disableCache()->whereDoesntHave('excludedServers', function ($q) use ($server) {
            $q->where('server', $server->id);
        })->first();

        if (!isset($role) && $server->role_sync_create) {
            $role = Role::updateOrCreate(
                ['name' => ucfirst($params['role']) . ' - ' . $server->name],
                ['ingame_equivalent' => $params['role']]
            );
            $role->flushCache();
        }

        if (isset($role))
            $user->addRole($role->rid);

        return $response->withJSON(['status' => 'success']);
    }

    function deleteUserRole($request, $response, $args) {
        $this->determineIdColumn($request);
        $user = User::where($this->idColumn, $args['id'])->first();
        $role = Role::where(['ingame_equivalent' => $args['ingame_equivalent']])->disableCache()->whereDoesntHave('excludedServers', function ($q) use ($request) {
            $q->where('server', $request->getAttribute('server')->id);
        })->first();
        $user->removeRole($role->rid);
        return $response->withJSON(['status'=>'success']);
    }

    /**
     * A role sync method used by game servers which only support a single group per player.
     */
    function postUserGroup($request, $response, $args) {
        $this->determineIdColumn($request);
        $user = User::where($this->idColumn, $args['id'])->first();
        $server = $request->getAttribute('server');

        $group = $request->getParam('group');
        if ($group == "user") $group = null;

        if ($group != null) {
            $role = Role::where(['ingame_equivalent' => $group])->disableCache()->whereDoesntHave('excludedServers', function ($q) use ($server) {
                $q->where('server', $server->id);
            })->first();

            if (!isset($role) && $server->role_sync_create) {
                $role = Role::updateOrCreate(
                    ['name' => ucfirst($group) . ' - ' . $server->name],
                    ['ingame_equivalent' => $group]
                );
                $role->flushCache();
            }

            if (isset($role))
                $user->addRole($role->rid);
        }

        $user->removeRoles(Role::disableCache()->whereDoesntHave('excludedServers', function ($q) use ($server) {
            $q->where('server', $server->id);
        })->where('order', '<', $role->order ?? 0)->get());

        return $response->withJSON(['status'=>'success']);
    }

    /**
     * A token-based account linking method for games which don't provide OAuth.
     */
    function postUserLink($request, $response, $args) {
        $params = $request->getParams();
        $this->determineIdColumn($request);
        $user = User::where($this->idColumn, $args['id'])->first();

        if (empty($params['token']))
            return $response->withJSON(['status' => 'failure']);

        $payload = Helpers::jwt()->decode($params['token']);
        $targetUser = User::find($payload['auth_user_id'] ?? null);

        if (!$targetUser || $targetUser->{$this->idColumn} !== null)
            return $response->withJSON(['status' => 'failure']);

        if (!$user->merge($targetUser))
            return $response->withJSON(['status' => 'failure']);

        return $response->withJSON(['status' => 'success']);
    }
}
